import torch

from attacks import Attack
import torch.nn.functional as F

from constants import DEVICE
from utils import cross_entropy_loss, de_normalization, normalization


class EPNSIM(Attack):
    """ EPN + SIM """

    def __init__(self, model, eps=16 / 255, steps=10, decay=1.0, epochs=5, m=5):
        """
        :param model: DNN model
        :param eps: the maximum perturbation
        :param steps: the number of iterations
        :param decay: the decay factor
        """
        super().__init__("EPNSIM", model)
        self.eps = eps
        self.steps = steps
        self.alpha = self.eps / self.steps
        self.decay = decay
        self.epochs = epochs
        self.m = m

    def get_ensemble_grad(self, x, targets):
        ensemble_grad = torch.zeros_like(x)
        for i in range(self.m):
            x_scaled = x / 2 ** i
            logits = self.model(x_scaled)
            loss = cross_entropy_loss(logits, targets)
            ensemble_grad += torch.autograd.grad(loss, x)[0]
        ensemble_grad /= self.m
        return ensemble_grad

    def forward(self, images, labels):
        targets = F.one_hot(labels.type(torch.int64), 1000).float().to(DEVICE)
        images_de_normalized = de_normalization(images)
        images_min = torch.clamp(images_de_normalized - self.eps, min=0.0, max=1.0)
        images_max = torch.clamp(images_de_normalized + self.eps, min=0.0, max=1.0)

        g = torch.zeros_like(images)
        for _ in range(self.epochs):
            adv_hat = images.clone()
            for _ in range(self.steps):
                adv_hat_exchanged = torch.stack([x[torch.randperm(3), :, :] for x in adv_hat])

                grad_hat = torch.zeros_like(images)
                for i in range(self.m):
                    adv_hat_exchanged_scaled = adv_hat_exchanged / 2 ** i
                    logits_scaled = self.model(adv_hat_exchanged_scaled)
                    loss_scaled = cross_entropy_loss(logits_scaled, targets)
                    grad_hat += torch.autograd.grad(loss_scaled, adv_hat, retain_graph=True)[0]
                grad_hat /= self.m

                adv_hat_de_normalized = de_normalization(adv_hat)
                adv_wave_de_normalized = adv_hat_de_normalized + self.alpha * grad_hat / torch.mean(torch.abs(grad_hat),
                                                                                                    dim=(1, 2, 3),
                                                                                                    keepdim=True) + self.decay * self.alpha * g
                adv_wave = normalization(adv_wave_de_normalized)
                grad_wave = self.get_ensemble_grad(adv_wave, targets)

                # adv_wave_exchanged = torch.stack([x[torch.randperm(3), :, :] for x in adv_wave])
                # grad_wave = torch.zeros_like(images)
                # for i in range(self.m):
                #     adv_wave_exchanged_scaled = adv_wave_exchanged / 2
                #     logits_scaled = self.model(adv_wave_exchanged_scaled)
                #     loss_scaled = cross_entropy_loss(logits_scaled, targets)
                #     grad_wave += torch.autograd.grad(loss_scaled, adv_wave, retain_graph=True)[0]
                # grad_wave /= self.m

                g = self.decay * g + grad_hat / torch.mean(torch.abs(grad_hat), dim=(1, 2, 3),
                                                           keepdim=True) + grad_wave / torch.mean(torch.abs(grad_wave),
                                                                                                  dim=(1, 2, 3),
                                                                                                  keepdim=True)

                adv_hat_de_normalized = de_normalization(adv_hat)
                adv_hat_de_normalized = torch.clamp(adv_hat_de_normalized + self.alpha * torch.sign(g), min=images_min,
                                                    max=images_max)
                adv_hat = normalization(adv_hat_de_normalized)

        adv = images.clone()
        for _ in range(self.steps):
            grad = self.get_ensemble_grad(adv, targets)

            adv_de_normalized = de_normalization(adv)
            adv_de_normalized = adv_de_normalized + self.alpha * grad / torch.mean(torch.abs(grad), dim=(1, 2, 3),
                                                                                   keepdim=True) + self.decay * self.alpha * g
            pre_adv = normalization(adv_de_normalized)

            pre_grad = self.get_ensemble_grad(pre_adv, targets)

            g = self.decay * g + grad / torch.mean(torch.abs(grad), dim=(1, 2, 3),
                                                   keepdim=True) + pre_grad / torch.mean(torch.abs(pre_grad),
                                                                                         dim=(1, 2, 3), keepdim=True)

            adv_de_normalized = de_normalization(adv)
            adv_de_normalized = torch.clamp(adv_de_normalized + self.alpha * torch.sign(g), min=images_min,
                                            max=images_max)
            adv = normalization(adv_de_normalized)

        return adv
